/*
    1.0 - 22.12.2022
 */


#define INTERVAL_ZAPISU_SD 5 //interval zapisu na SD kartu - v minutach


//struktura pro ulozeni online dat, ktere se posilaji do grafu - neukladaji se na SD kartu
struct ChartItem {
    unsigned int timeStamp;
    int value;
};
struct ChartItem chartStack[80];
unsigned int chartStackSize = 0;


//systemove promenne
unsigned int lastTimeTimer[2];
unsigned int lastTimeModbus;
unsigned int lastTimeSdCard;

unsigned char serialBuffer[256]; //buffer pro Rs485
char fileBuffer[64]; //buffer pro zapis do souboru

//sdilene promenne, ktere se prenasi na web
char S01[512];
char S02[512];
char S03[1024];


//funkce pro pridani aktualniho vykonu do pameti - prenasi se potom na web do grafu, neuklada se na SD kartu
//funguje to jako fronta - v pameti se drzi poslednich cca 80 hodnot. Pokud je fronta naplnena, posune se cela o jednu pozici a z fronty vypadne nejstarsi polozka
void addChartStackItem(float floatValue) {
    unsigned int timeStamp, index, a, maxSize, startTime;
    int value;

    //neni jeste synchronizovano s NTP
    if(getNtpState() == 0) return;

    startTime = millis();
    value = (int) (floatValue * 1000);

    //NTP true raw epoch-time
    timeStamp = epochTime();
    
    if(chartStackSize == 0) {
        chartStack[chartStackSize].timeStamp = timeStamp;
        chartStack[chartStackSize].value = value;
        chartStackSize++;
    }
    else {
        //posun hodnot v poli
        if(chartStackSize >= 80) {
            for(a = 0; a < 80-1; a++) {
                chartStack[a].timeStamp = chartStack[a + 1].timeStamp;
                chartStack[a].value = chartStack[a + 1].value;
            }
        } 
        else {
            chartStackSize++;
        }

        chartStack[chartStackSize-1].timeStamp = timeStamp;
        chartStack[chartStackSize-1].value = value;
    }

    index = 0;
    maxSize = sizeof(S03) - 1; //minus -1 je ukoncovaci znak 0x00

    for(a = 0; a < chartStackSize; a++) {
        if(a > 0) {
            index += snprintf(&S03[index], max(0, maxSize-index), "&");
        }

        index += snprintf(&S03[index], max(0, maxSize-index), "%x=%i", chartStack[a].timeStamp - 1670000000, chartStack[a].value);
    }
}


void saveDataSdCard(float total, float odber, float dodano) {
    unsigned int bytesWritten, startTime, filePtr, intervalZapisu;
    int vysledek;

    //neni jeste synchronizovano s NTP
    if(getNtpState() == 0) return;
    
    intervalZapisu = 60000 * max(1, INTERVAL_ZAPISU_SD);
    
    if((unsigned int) (millis() - lastTimeSdCard) >= intervalZapisu) {
        startTime = millis();

        if(openFile(&filePtr) == 0) {
            sprintf(fileBuffer, "%02u.%02u.%u %02u:%02u:%02u;%0.3f;%0.3f;%0.3f\r\n", SDS_get_u32(5), SDS_get_u32(6), SDS_get_u32(7), SDS_get_u32(8), SDS_get_u32(9), SDS_get_u32(10), total, odber, dodano);                   

            vysledek = file_write(filePtr, fileBuffer, strlen(fileBuffer), &bytesWritten);
            if(vysledek == 0) {
                lastTimeSdCard = millis();
            }
            closeFile(filePtr);

            echoTime();
            printf("SD zapis %0.3f, %0.3f, %0.3f, vysledek=%i, bytes=%u, time=%ums\n", total, odber, dodano, vysledek, bytesWritten, (millis() - startTime));
        }
    }
}





//Nacte aktualni hodnoty vykonu, proudu, napeti a frekvence z elektromeru
void pro380MbOnlineData() {
    unsigned int modStatus, index, startTime;
    float vykonCelkem;

    modStatus = readRS485Data(1, 0x5002, 24, 1000);
    startTime = millis();
    
    index = 0;
    index += sprintf(&S01[index], "mbs=%i&", modStatus);
    index += sprintf(&S01[index], "dtb=%u&", (millis() - lastTimeModbus)/1000); //pred kolika vterinama prisly posledni data z elektromeru
    index += sprintf(&S01[index], "ntp=%u&", getNtpState()); //Stav NTP synchronizace
    index += sprintf(&S01[index], "sdc=%u&", SDS_get_u32(3850)); //Stav SD karty
    index += sprintf(&S01[index], "sdt=%u&", SDS_get_u32(3853)); //Typ SD karty
    index += sprintf(&S01[index], "sdv=%u&", SDS_get_u32(3857)); //Velikost SD karty  
    
    //cteni bylo uspesne    
    if(modStatus == 0) {
        lastTimeModbus = millis();

        vykonCelkem = getFloat(&serialBuffer, 35);

        //vlozeni hodnot z elektromeru do sdilene promenne
        index += sprintf(&S01[index], "u1=%0.1f&", getFloat(&serialBuffer, 3));
        index += sprintf(&S01[index], "u2=%0.1f&", getFloat(&serialBuffer, 7));
        index += sprintf(&S01[index], "u3=%0.1f&", getFloat(&serialBuffer, 11));
        index += sprintf(&S01[index], "f=%0.1f&", getFloat(&serialBuffer, 15));

        index += sprintf(&S01[index], "i1=%0.3f&", getFloat(&serialBuffer, 23));
        index += sprintf(&S01[index], "i2=%0.3f&", getFloat(&serialBuffer, 27));
        index += sprintf(&S01[index], "i3=%0.3f&", getFloat(&serialBuffer, 31));

        index += sprintf(&S01[index], "pt=%0.3f&", vykonCelkem);
        index += sprintf(&S01[index], "p1=%0.3f&", getFloat(&serialBuffer, 39));
        index += sprintf(&S01[index], "p2=%0.3f&", getFloat(&serialBuffer, 43));
        index += sprintf(&S01[index], "p3=%0.3f&", getFloat(&serialBuffer, 47));       

        //vlozeni dat do pameti pro graf
        addChartStackItem(vykonCelkem);
    }    

    sprintf(&S01[index], "t=%u", (millis() - startTime));
}

//Nacteni pocitadel (dodavka, odber, celkem) z elektromeru
void pro380MbCounterData() {
    unsigned int modStatus, index, startTime;

    float total, totalVt, totalNt;
    float odber, odberVt, odberNt;
    float dodano, dodanoVt, dodanoNt;

    startTime = millis();

    modStatus = readRS485Data(1, 0x6000, 36, 1000);

    index = 0;
    index += sprintf(&S02[index], "mbs=%i&", modStatus);

    //cteni bylo uspesne    
    if(modStatus == 0) {
        total = getFloat(&serialBuffer, 3);
        totalVt = getFloat(&serialBuffer, 7);
        totalNt = getFloat(&serialBuffer, 11);
        
        index += sprintf(&S02[index], "tt=%.2f&", total);
        index += sprintf(&S02[index], "tv=%.2f&", totalVt);
        index += sprintf(&S02[index], "tn=%.2f&", totalNt);
        index += sprintf(&S02[index], "t1=%.2f&", getFloat(&serialBuffer, 15));
        index += sprintf(&S02[index], "t2=%.2f&", getFloat(&serialBuffer, 19));
        index += sprintf(&S02[index], "t3=%.2f&", getFloat(&serialBuffer, 23));


        odber = getFloat(&serialBuffer, 27);
        odberVt = getFloat(&serialBuffer,31);
        odberNt = getFloat(&serialBuffer, 35);

        index += sprintf(&S02[index], "ot=%.2f&", odber);
        index += sprintf(&S02[index], "ov=%.2f&", odberVt);
        index += sprintf(&S02[index], "on=%.2f&", odberNt);
        index += sprintf(&S02[index], "o1=%.2f&", getFloat(&serialBuffer, 39));
        index += sprintf(&S02[index], "o2=%.2f&", getFloat(&serialBuffer, 43));
        index += sprintf(&S02[index], "o3=%.2f&", getFloat(&serialBuffer, 47));


        dodano = getFloat(&serialBuffer, 51);
        dodanoVt = getFloat(&serialBuffer, 55);
        dodanoNt = getFloat(&serialBuffer, 59);

        index += sprintf(&S02[index], "dt=%.2f&", dodano);
        index += sprintf(&S02[index], "dv=%.2f&", dodanoVt);
        index += sprintf(&S02[index], "dn=%.2f&", dodanoNt);
        index += sprintf(&S02[index], "d1=%.2f&", getFloat(&serialBuffer, 63));
        index += sprintf(&S02[index], "d2=%.2f&", getFloat(&serialBuffer, 67));
        index += sprintf(&S02[index], "d3=%.2f&", getFloat(&serialBuffer, 71)); 

        //zapis na SD kartu
        saveDataSdCard(total, odber, dodano);
    }    
    
    sprintf(&S02[index], "t=%u", (millis() - startTime));
}


//Prevede cislo slozene ze 4 bajtu na float (desetinne cislo)
float getFloat(unsigned char *buffer, int index) {
    unsigned int tmpCislo;

    tmpCislo = ((buffer[index] << 24) | (buffer[index+1] << 16) | (buffer[index+2] << 8) | buffer[index+3]);
    return (*(float *) &tmpCislo);
}


//Obecna funkce pro cteni dat pres Modbus protokol
unsigned int readRS485Data(int slaveId, int startAddr, int count, int timeOut) {
    unsigned int startTime, bufIndex;
    int bytesLeft, status, pocet, crcData, crc;
    unsigned char znak;
    
    bufIndex = 0;
    serialBuffer[bufIndex++] = slaveId;
    serialBuffer[bufIndex++] = 0x03;
    serialBuffer[bufIndex++] = (startAddr >> 8) & 0xFF;
    serialBuffer[bufIndex++] = startAddr & 0xFF;
    serialBuffer[bufIndex++] = (count >> 8) & 0xFF;
    serialBuffer[bufIndex++] = count & 0xFF;
    
    crc = getBufferCrc16(&serialBuffer, bufIndex);
    serialBuffer[bufIndex++] = crc & 0xFF;
    serialBuffer[bufIndex++] = (crc >> 8) & 0xFF;
    
    //vycisteni prijimaciho buffer
    while(SDS_serial_read(1, (void *) &znak, 1) > 0) {}

    SDS_serial_write(1, (void *) serialBuffer, bufIndex);

    startTime = millis();
    
    status = 0;
    bytesLeft = (count*2) + 5; //2x bajty + slaveId + 2xdelka
    bufIndex = 0;
    
    while(bytesLeft && !status) {
        pocet = SDS_serial_read(1, (void *) &znak, 1);
        if(pocet > 0) {   
            bytesLeft--;
            serialBuffer[bufIndex++] = znak;
        }
        
        //timeOut
        if(millis() - startTime > timeOut) {
            status = 1;
        }
    }

    if(status == 0) {
        crcData = getBufferCrc16(&serialBuffer, bufIndex - 2);
        crc = ((serialBuffer[bufIndex - 1]) << 8) | serialBuffer[bufIndex - 2];

        //nesouhlasi CRC
        if(crcData != crc) {
            status = 2;
        }
    }

    return status;
}

//Vypocet CRC hondoty dat posilanych a prijimanych pres Modbus
int getBufferCrc16(unsigned char *buffer, int length) {
    int crc = 0xFFFF;

    for(int i = 0; i < length; i++) {
        crc ^= buffer[i] & 0xFF;

        for(int j = 8; j != 0; j--) {
            if((crc & 0x0001) != 0) {
                 crc = (crc >> 1) ^ 0xA001;
            } else {
                 crc = crc >> 1;
            }
        }
    }
    
    return crc & 0xFFFF;
}


void closeFile(unsigned int filePtr) {
    file_close(&filePtr);
}

//funkce pro otvereni (vytvoreni) souboru pro aktualni den - pro zapis hodnot
int openFile(unsigned int *filePtr) {
    uint64_t size;
    int vysledek;
   
    sprintf(fileBuffer, "web/data/%04u-%02u-%02u.csv", SDS_get_u32(7), SDS_get_u32(6), SDS_get_u32(5));                   
    
    vysledek = file_open(&(*filePtr), fileBuffer, "a");
    if(vysledek != 0) {
        echoTime();
        printf("Soubor %s se nepodarilo otevrit, chyba: %i\n", fileBuffer, vysledek);

        return -1;  //chyba
    }
    
    size = file_size(*filePtr);  //velikost souboru
    
    file_lseek(*filePtr, size);  //ukazatel na konec souboru

    echoTime();    
    printf("Soubor '%s' otevren, velikost: %ubytes, filePtr: %u\n", fileBuffer, size, *filePtr);

    return 0;
}





//vypise datum a cas do HTML stranky
void echoTime() {
    printf("[%u.%u.%u %02u:%02u:%02u] ", SDS_get_u32(5), SDS_get_u32(6), SDS_get_u32(7), SDS_get_u32(8), SDS_get_u32(9), SDS_get_u32(10));
}

//Vraci nizsi hodnotu dvou cisel
int min(int x, int y) {
    if(x < y) return x;
    else return y;
}

//Vraci vyssi hodnotu dvou cisel
int max(int x, int y) {
    if(x > y) return x;
    else return y;
}

//vraci hodnotu epochTime - RAW hodnota bez offsetu a posunu
unsigned int epochTime() {
    return SDS_get_u32(139);
}

//vraci hodnotu upTime (cas od spusteni SDS)
unsigned int millis() {
    return SDS_get_u32(45);
}

//Vraci stav NTP synchronizace
//0 - neni, ostatni - je OK
unsigned int getNtpState() {
    return SDS_get_u32(44);
}

//Metoda init() se zavola jen 1x pri spuseni programu
void init() {
    
    //konfigurace pripojeni k RS485 - vychozi hodnoty pro elektromer
    SDS_serial_config(1, 9600, 9, 1, 2);
}
    


//nekonecna smycka programu
void loop() {
    if((unsigned int) (millis() - lastTimeTimer[0]) >= 5000) { //spusti se co 5 sekund
        lastTimeTimer[0] = millis();

        pro380MbOnlineData();
    }

    if((unsigned int) (millis() - lastTimeTimer[1]) >= 30000) { //spusti se co 30 sekund
        lastTimeTimer[1] = millis();

        pro380MbCounterData();
    }
}

void main(void) {
    init();
   
    while(1) {
        loop();
    }
}

